<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Kyc;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Gate;
use Illuminate\Support\Facades\Log;

class KycFileController extends Controller
{
    public function download(Kyc $kyc, string $type)
    {
        try {
            // Check authorization
            Gate::authorize('process kyc applications');
            
            // Validate type parameter
            if (!in_array($type, ['front', 'back'])) {
                abort(400, 'Invalid file type');
            }
            
            // Get the appropriate file path
            $dbFilePath = null;
            if ($type === 'front') {
                $dbFilePath = $kyc->frontimg;
            } elseif ($type === 'back') {
                $dbFilePath = $kyc->backimg;
            }
            
            // Check if file path exists in database
            if (empty($dbFilePath)) {
                abort(404, 'File path not found in database');
            }
            
            // Try multiple possible file locations
            $possiblePaths = [];
            
            // New assets location
            if (str_starts_with($dbFilePath, 'assets/')) {
                $possiblePaths[] = public_path($dbFilePath);
            } else {
                $possiblePaths[] = public_path('assets/kyc/' . basename($dbFilePath));
            }
            
            // Old storage locations (fallback)
            $possiblePaths[] = storage_path('app/' . $dbFilePath);
            $possiblePaths[] = storage_path('app/public/' . $dbFilePath);
            
            $filePath = null;
            foreach ($possiblePaths as $path) {
                if (file_exists($path)) {
                    $filePath = $path;
                    break;
                }
            }
            
            // Log for debugging
            Log::info('KYC File Download Attempt', [
                'kyc_id' => $kyc->id,
                'type' => $type,
                'db_path' => $dbFilePath,
                'possible_paths' => $possiblePaths,
                'found_path' => $filePath
            ]);
            
            // Check if file exists
            if (!$filePath) {
                abort(404, 'File not found in any location. Checked: ' . implode(', ', $possiblePaths));
            }
            
            // Return file download
            return response()->download($filePath);
            
        } catch (\Illuminate\Auth\Access\AuthorizationException $e) {
            abort(403, 'Unauthorized access');
        } catch (\Throwable $th) {
            Log::error('KYC File Download Error', [
                'kyc_id' => $kyc->id ?? 'unknown',
                'type' => $type,
                'error' => $th->getMessage(),
                'trace' => $th->getTraceAsString()
            ]);
            abort(500, 'Error downloading file: ' . $th->getMessage());
        }
    }
}