<?php

namespace App\Http\Controllers;

use App\Models\Message;
use App\Models\User;
use App\Events\NewMessage;
use App\Events\UserTyping;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\Log;

class ChatController extends Controller
{
    public function getUserMessages()
    {
        try {
            return Message::where(function($query) {
                $query->where('from_user', auth()->id())
                    ->orWhere('to_user', auth()->id());
            })
            ->with(['fromUser:id,name,is_admin'])
            ->orderBy('created_at', 'ASC')
            ->get()
            ->map(function ($message) {
                $message->from_user_is_admin = $message->fromUser->is_admin;
                return $message;
            });
        } catch (\Exception $e) {
            Log::error('Error in getUserMessages: ' . $e->getMessage());
            return response()->json(['error' => 'Failed to get messages'], 500);
        }
    }

    public function sendUserMessage(Request $request)
	{
    try {
        // Get the first admin user
        $admin = User::where('is_admin', 1)->first();
        
        if (!$admin) {
            return response()->json(['error' => 'No admin available'], 404);
        }

        if (!$request->message && !$request->hasFile('file')) {
            return response()->json(['error' => 'Message or file is required'], 422);
        }

        $message = new Message();
        $message->from_user = auth()->id();
        $message->to_user = $admin->id;
        $message->message = $request->message ?? '';
        $message->read = false;

        // Handle file upload
        if ($request->hasFile('file')) {
            $file = $request->file('file');
            
            // Validate file
            $request->validate([
                'file' => 'file|max:10240|mimes:jpeg,png,gif,pdf,doc,docx'
            ]);

            $path = $file->store('chat_files', 'public');
            $message->file_url = Storage::url($path);
            $message->file_name = $file->getClientOriginalName();
            $message->file_type = $file->getMimeType();
        }

        $message->save();

        broadcast(new NewMessage($message))->toOthers();

        return response()->json($message);
    } catch (\Exception $e) {
        Log::error('Error in sendUserMessage: ' . $e->getMessage());
        return response()->json(['error' => 'Failed to send message'], 500);
    }
	}

	public function endChat()
	{
    try {
        // Get all messages involving this user
        $messages = Message::where('from_user', auth()->id())
            ->orWhere('to_user', auth()->id())
            ->get();

        // Delete associated files
        foreach ($messages as $message) {
            if ($message->file_url) {
                // Get file path from URL
                $path = str_replace('/storage/', '', $message->file_url);
                if (Storage::disk('public')->exists($path)) {
                    Storage::disk('public')->delete($path);
                }
            }
        }

        // Delete all messages
        Message::where('from_user', auth()->id())
            ->orWhere('to_user', auth()->id())
            ->delete();

        return response()->json([
            'success' => true,
            'message' => 'Chat ended and all messages deleted successfully'
        ]);

    } catch (\Exception $e) {
        Log::error('Error ending chat: ' . $e->getMessage());
        return response()->json([
            'success' => false,
            'error' => 'Failed to end chat'
        ], 500);
    }
	}

    public function typing()
    {
        try {
            broadcast(new UserTyping(auth()->id()))->toOthers();
            return response()->json(['success' => true]);
        } catch (\Exception $e) {
            Log::error('Error in typing: ' . $e->getMessage());
            return response()->json(['error' => 'Failed to broadcast typing event'], 500);
        }
    }

    public function getUnreadCount()
    {
        try {
            $count = Message::where('to_user', auth()->id())
                ->where('read', false)
                ->count();
            return response()->json(['count' => $count]);
        } catch (\Exception $e) {
            Log::error('Error in getUnreadCount: ' . $e->getMessage());
            return response()->json(['error' => 'Failed to get unread count'], 500);
        }
    }

    public function markAsRead()
    {
        try {
            Message::where('to_user', auth()->id())
                ->where('read', false)
                ->update(['read' => true]);
            return response()->json(['success' => true]);
        } catch (\Exception $e) {
            Log::error('Error in markAsRead: ' . $e->getMessage());
            return response()->json(['error' => 'Failed to mark messages as read'], 500);
        }
    }
    
    public function updateStatus()
    {
        try {
            $user = auth()->user();
            $user->update([
                'is_online' => true,
                'last_seen' => now()
            ]);
            return response()->json(['success' => true]);
        } catch (\Exception $e) {
            Log::error('Error in updateStatus: ' . $e->getMessage());
            return response()->json(['error' => 'Failed to update status'], 500);
        }
    }
}