<?php

namespace App\Jobs;

use App\Models\Trade;
use App\Models\User;
use App\Models\CopyTrade;
use App\Models\CopiedTrade;
use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Foundation\Bus\Dispatchable;
use Illuminate\Queue\InteractsWithQueue;
use Illuminate\Queue\SerializesModels;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Log;

class ProcessTrade implements ShouldQueue
{
    use Dispatchable, InteractsWithQueue, Queueable, SerializesModels;

    public $tries = 3;
    public $timeout = 30;

    public function __construct(protected Trade $trade)
    {
    }

    public function handle(): void
{
    try {
        $currentPrice = null;
        $priceDifference = 0;
        $priceAvailable = false;

        // Try to fetch current price from database first
        try {
            $symbol = strtoupper(str_replace(['/', 'USDT'], '', $this->trade->pair));
            $asset = \App\Models\CryptoAsset::where('symbol', $symbol)->first();

            if ($asset && $asset->current_price > 0) {
                $currentPrice = $asset->current_price;
                $priceAvailable = true;
            }
        } catch (\Exception $e) {
            Log::warning('Failed to fetch price from database, will use fallback logic', [
                'trade_id' => $this->trade->id,
                'error' => $e->getMessage()
            ]);
        }

        // If expected_result is set, use it instead of calculating
        if ($this->trade->expected_result) {
            $isWin = $this->trade->expected_result === 'win';
            
            if ($priceAvailable) {
                $priceDifference = (($currentPrice - $this->trade->entry_price) / $this->trade->entry_price) * 100;
            } else {
                // Simulate realistic price movement
                $priceDifference = ($isWin ? 1 : -1) * (rand(5, 50) / 100);
                $currentPrice = $this->trade->entry_price * (1 + ($priceDifference / 100));
            }

            // Calculate profit/loss based on expected_result
            $profitAmount = $isWin
                ? $this->trade->amount * ($this->trade->profit_percentage / 100)
                : -$this->trade->amount;

        } else {
            // Determine result based on database price or fallback
            if ($priceAvailable) {
                // Original logic for determining trade result using database price
                $result = $this->determineTradeResult(
                    $currentPrice,
                    $this->trade->entry_price,
                    $this->trade->amount,
                    $this->trade->duration,
                    $this->trade->type
                );

                $isWin = $result['isWin'];
                $priceDifference = $result['priceDifference'];
            } else {
                // Fallback: Use configured win rate to determine result
                $settings = \App\Models\Settings::first();
                $winProbability = $settings->profit_percent ?? 50; // Use profit_percent or default 50%
                
                $isWin = (rand(1, 100) <= $winProbability);
                
                // Simulate realistic price movement
                $priceDifference = ($isWin ? 1 : -1) * (rand(5, 50) / 100);
                $currentPrice = $this->trade->entry_price * (1 + ($priceDifference / 100));
                
                Log::info('Trade processed with fallback logic (price unavailable)', [
                    'trade_id' => $this->trade->id,
                    'result' => $isWin ? 'win' : 'loss',
                    'win_probability' => $winProbability
                ]);
            }

            $profitAmount = $isWin
                ? $this->trade->amount * ($this->trade->profit_percentage / 100)
                : -$this->trade->amount;
        }

        // Update trade
        $this->trade->status = 'completed';
        $this->trade->result = $isWin ? 'win' : 'loss';
        $this->trade->close_price = $currentPrice;
        $this->trade->profit_amount = $profitAmount;
        $this->trade->price_difference = $priceDifference;
        $this->trade->completed_at = now();
        $this->trade->save();

        // Handle user updates
        $user = $this->trade->user;
        if ($isWin) {
            $winAmount = $this->trade->amount + $profitAmount;
            $user->account_bal += $winAmount;
        }

        // Update user stats
        $user->total_trades += 1;
        if ($isWin) {
            $user->total_wins += 1;
        } else {
            $user->total_losses += 1;
        }
        $user->win_rate = $user->total_trades > 0 ? 
            round(($user->total_wins / $user->total_trades) * 100, 2) : 0;
        $user->save();

        // Handle copy trades if this was a trader's trade
        $copyTrades = CopyTrade::where('trader_id', $this->trade->user_id)
            ->where('status', 'active')
            ->get();

        foreach ($copyTrades as $copyTrade) {
            try {
                // Create a new trade for the follower
                $followerTrade = Trade::create([
                    'user_id' => $copyTrade->follower_id,
                    'pair' => $this->trade->pair,
                    'amount' => $copyTrade->amount_per_trade,
                    'type' => $this->trade->type,
                    'entry_price' => $this->trade->entry_price,
                    'profit_percentage' => $this->trade->profit_percentage,
                    'duration' => $copyTrade->duration,
                    'status' => 'completed',
                    'result' => $isWin ? 'win' : 'loss',
                    'close_price' => $currentPrice,
                    'price_difference' => $priceDifference,
                    'completed_at' => now()
                ]);

                // Calculate profit for follower's trade
                $followerProfitAmount = $isWin
                    ? $copyTrade->amount_per_trade * ($this->trade->profit_percentage / 100)
                    : -$copyTrade->amount_per_trade;

                $followerTrade->profit_amount = $followerProfitAmount;
                $followerTrade->save();

                // Create copied trade record
                $copiedTrade = CopiedTrade::create([
                    'copy_trade_id' => $copyTrade->id,
                    'original_trade_id' => $this->trade->id,
                    'copied_trade_id' => $followerTrade->id,
                    'amount' => $copyTrade->amount_per_trade,
                    'status' => 'completed',
                    'profit_amount' => $followerProfitAmount,
                    'result' => $isWin ? 'win' : 'loss',
                    'completed_at' => now()
                ]);

                // Update follower's balance if trade was successful
                $follower = User::find($copyTrade->follower_id);
                if ($isWin) {
                    $winAmount = $copyTrade->amount_per_trade + $followerProfitAmount;
                    $follower->account_bal += $winAmount;
                }

                // Update follower's stats
                $follower->total_trades += 1;
                if ($isWin) {
                    $follower->total_wins += 1;
                } else {
                    $follower->total_losses += 1;
                }
                $follower->win_rate = $follower->total_trades > 0 ? 
                    round(($follower->total_wins / $follower->total_trades) * 100, 2) : 0;
                $follower->save();

                // Update copy trade stats
                $copyTrade->updateStats();

                Log::info('Copy trade completed', [
                    'original_trade_id' => $this->trade->id,
                    'copied_trade_id' => $followerTrade->id,
                    'follower_id' => $copyTrade->follower_id,
                    'amount' => $copyTrade->amount_per_trade,
                    'profit_amount' => $followerProfitAmount,
                    'result' => $isWin ? 'win' : 'loss'
                ]);

            } catch (\Exception $e) {
                Log::error('Copy trade processing error', [
                    'copy_trade_id' => $copyTrade->id,
                    'original_trade_id' => $this->trade->id,
                    'follower_id' => $copyTrade->follower_id,
                    'error' => $e->getMessage()
                ]);

                // Refund follower if copy trade fails
                $follower = User::find($copyTrade->follower_id);
                $follower->increment('account_bal', $copyTrade->amount_per_trade);

                // Create failed copy trade record
                CopiedTrade::create([
                    'copy_trade_id' => $copyTrade->id,
                    'original_trade_id' => $this->trade->id,
                    'amount' => $copyTrade->amount_per_trade,
                    'status' => 'failed',
                    'error_message' => $e->getMessage()
                ]);
            }
        }

        // Log detailed trade result
        Log::info('Trade completed', [
            'trade_id' => $this->trade->id,
            'user_id' => $this->trade->user_id,
            'result' => $isWin ? 'win' : 'loss',
            'entry_price' => $this->trade->entry_price,
            'close_price' => $currentPrice,
            'price_difference' => $priceDifference,
            'profit_amount' => $profitAmount,
            'result_determination' => $this->trade->expected_result ? 'manual' : 'automatic',
            'expected_result' => $this->trade->expected_result
        ]);

    } catch (\Exception $e) {
        Log::error('Trade processing error', [
            'trade_id' => $this->trade->id,
            'error' => $e->getMessage(),
            'trace' => $e->getTraceAsString()
        ]);

        // Handle refund
        $this->trade->status = 'refunded';
        $this->trade->result = 'error';
        $this->trade->completed_at = now();
        $this->trade->save();

        $user = $this->trade->user;
        $user->account_bal += $this->trade->amount;
        $user->save();

        Log::info('Trade refunded due to error', [
            'trade_id' => $this->trade->id,
            'user_id' => $this->trade->user_id,
            'amount' => $this->trade->amount
        ]);

        throw $e;
    }
}

    private function determineTradeResult(float $currentPrice, float $entryPrice, float $amount, string $duration, string $type): array
    {
        // Calculate base price difference percentage
        $priceDifference = (($currentPrice - $entryPrice) / $entryPrice) * 100;
        
        // Dynamic threshold based on amount tiers
        $amountThreshold = match(true) {
            $amount < 10 => 0.01,  
            $amount < 50 => 0.02,
            $amount < 100 => 0.05,
            $amount < 500 => 0.08,
            $amount < 1000 => 0.1,
            default => 0.15
        };

        // Duration modifier (shorter duration = lower threshold)
        $durationModifier = match($duration) {
            '1m' => 0.5,
            '5m' => 0.6,
            '15m' => 0.7,
            '30m' => 0.8,
            '1h' => 0.9,
            '4h' => 1.0,
            '1d' => 1.2,
            default => 1.0
        };

        // Get market volatility
        $volatility = $this->calculateMarketVolatility($this->trade->pair);
        
        // Final threshold calculation
        $finalThreshold = $amountThreshold * $durationModifier * (1 / max($volatility, 0.1));

        // Determine win/loss with final threshold
        $isWin = match($type) {
            'Rise' => $priceDifference >= $finalThreshold,
            'Fall' => $priceDifference <= -$finalThreshold,
            default => false
        };

        return [
            'isWin' => $isWin,
            'priceDifference' => $priceDifference,
            'appliedThreshold' => $finalThreshold,
            'metadata' => [
                'baseThreshold' => $amountThreshold,
                'durationModifier' => $durationModifier,
                'volatilityFactor' => $volatility,
            ]
        ];
    }

    private function calculateMarketVolatility(string $pair): float
    {
        try {
            // Get recent candles from Huobi for volatility calculation
            $response = Http::timeout(3)->get('https://api.huobi.pro/market/history/kline', [
                'symbol' => strtolower(str_replace('/', '', $pair)),
                'period' => '1min',
                'size' => 10
            ]);

            if (!$response->successful() || $response->json()['status'] !== 'ok') {
                return 1.0;
            }

            $candles = $response->json()['data'];
            
            // Calculate average price movement
            $movements = [];
            for ($i = 1; $i < count($candles); $i++) {
                $movement = abs(($candles[$i]['close'] - $candles[$i-1]['close']) / $candles[$i-1]['close'] * 100);
                $movements[] = $movement;
            }

            // Calculate volatility factor
            $avgMovement = array_sum($movements) / count($movements);
            
            // Normalize volatility (0.5 to 2.0 range)
            return min(max($avgMovement / 0.05, 0.5), 2.0);

        } catch (\Exception $e) {
            Log::warning('Failed to calculate market volatility', [
                'pair' => $pair,
                'error' => $e->getMessage()
            ]);
            return 1.0;
        }
    }

    public function failed(\Throwable $exception): void
    {
        Log::error('Trade job failed', [
            'trade_id' => $this->trade->id,
            'error' => $exception->getMessage(),
            'trace' => $exception->getTraceAsString()
        ]);

        // Ensure refund on complete failure
        if ($this->trade->status !== 'refunded') {
            $this->trade->status = 'refunded';
            $this->trade->result = 'error';
            $this->trade->completed_at = now();
            $this->trade->save();

            $user = $this->trade->user;
            $user->account_bal += $this->trade->amount;
            $user->save();
        }
    }
}