<?php

namespace App\Livewire\Admin;

use App\Models\User;
use Illuminate\Support\Facades\Hash;
use Illuminate\Validation\Rules\Password;
use Jantinnerezo\LivewireAlert\LivewireAlert;
use Livewire\Attributes\Layout;
use Livewire\Component;

#[Layout('layouts.admin')]
class AccountSettings extends Component
{
    use LivewireAlert;

    public User $user;
    public $name;
    public $phone_number;
    public $enable_2fa;
    public $current_password;
    public $password;
    public $password_confirmation;
    public $current_email;
    public $new_email;
    public $email_password;
    public function mount(): void
    {
        $this->user = User::find(auth()->user()->id);

        $this->name = $this->user->name;
        $this->phone_number = $this->user->phone_number;
        $this->enable_2fa = $this->user->enable_2fa;
        $this->current_email = $this->user->email;
    }
    public function render()
    {
        return view('livewire.admin.account-settings');
    }

    public function save(): void
    {
        $val = $this->validate([
            'name' => ['required', 'max:190'],
            'phone_number' => ['required', 'max:190'],
            'enable_2fa' => ['required'],
        ]);

        $this->user->update($val);

        $this->alert(message: __('Account settings updated successfully.'));
    }

    public function changePassword(): void
    {
        try {
            // Manual validation for current password
            $this->validate([
                'current_password' => ['required'],
                'password' => ['required'],
            ]);

            // Check if current password matches
            if (!Hash::check($this->current_password, $this->user->password)) {
                $this->addError('current_password', 'The current password is incorrect.');
                return;
            }

            $this->user->update(['password' => Hash::make($this->password)]);

            $this->reset(['current_password', 'password', 'password_confirmation']);
            $this->alert(type: 'success', message: 'Password changed successfully.');
            
        } catch (\Illuminate\Validation\ValidationException $e) {
            // Re-throw validation exception to show errors
            throw $e;
        } catch (\Exception $e) {
            $this->alert(type: 'error', message: 'An error occurred while changing password: ' . $e->getMessage());
        }
    }

    public function changeEmail(): void
    {
        try {
            // Validate input
            $this->validate([
                'new_email' => ['required', 'email', 'unique:users,email,' . $this->user->id],
                'email_password' => ['required'],
            ]);

            // Check if password is correct
            if (!Hash::check($this->email_password, $this->user->password)) {
                $this->addError('email_password', 'The password is incorrect.');
                return;
            }

            // Update email
            $this->user->update(['email' => $this->new_email]);
            $this->current_email = $this->new_email;

            $this->reset(['new_email', 'email_password']);
            $this->alert(type: 'success', message: 'Email address changed successfully.');
            
        } catch (\Illuminate\Validation\ValidationException $e) {
            // Re-throw validation exception to show errors
            throw $e;
        } catch (\Exception $e) {
            $this->alert(type: 'error', message: 'An error occurred while changing email: ' . $e->getMessage());
        }
    }
}
