<?php


namespace App\Livewire\User\Membership;

use App\Exceptions\MembershipException;
use App\Mail\MembershipMail;
use App\Models\Settings;
use App\Models\Transaction;
use App\Models\User;
use App\Notifications\User\MembershipCoursePurchased;
use App\Services\MembershipService;
use Illuminate\Support\Facades\Mail;
use Livewire\Component;

class CourseDetails extends Component
{
    public $data;
    public $purchased = false;

    public function mount(MembershipService $membership, string $course, string $id)
    {
        $settings = Settings::select('modules')->find(1);
        
        $isMembershipEnabled = $settings 
            && is_array($settings->modules) 
            && ($settings->modules['membership'] === 'true' || $settings->modules['membership'] === true);
        
        if (!$isMembershipEnabled) {
            session()->flash('message', 'Membership module is currently disabled.');
            return redirect()->route('user.dashboard');
        }

        try {
            $this->data = $membership->course($id);
            $this->purchased = in_array(auth()->user()->id, $this->data->users->pluck('id')->toArray());
        } catch (MembershipException $e) {
            session()->flash('message', $e->getMessage());
            return redirect()->route('user.dashboard');
        } catch (\Throwable $e) {
            session()->flash('error', 'An unexpected error occurred. Please try again.');
            return redirect()->route('user.dashboard');
        }
    }

    public function render()
    {
        $template = Settings::select('theme')->find(1)->theme ?? 'default';
        return view("{$template}.membership.course-details")
            ->extends("layouts.{$template}")
            ->title('Course Details');
    }

   public function buyCourse(MembershipService $membership)
{
    $user = auth()->user();
    
    if ($user->account_bal < $this->data->price) {
        session()->flash('message', __('t.course_insufficient_balance'));
        return;
    }

    try {
        $response = $membership->buyCourse($user->id, $this->data->id);
        $course_name = $this->data->title;

        // Create transaction record
        Transaction::create([
            'user_id' => $user->id,
            'amount' => $this->data->price,
            'narration' => __('t.course_transaction_narration', ['course_name' => $course_name]),
            'type' => 'Debit',
        ]);

        // Fetch settings once to reduce database queries
        $setting = Settings::select(['send_buy_course_email', 'receive_buy_course_email', 'notifiable_email'])->find(1);

        // Send user notification if enabled
        if ($setting->send_buy_course_email) {
            dispatch(function () use ($user, $course_name) {
                $user->notify(new MembershipCoursePurchased(
                    $course_name, 
                    $this->data->price, 
                    __('t.course_email_sent_user', ['course_name' => $course_name])
                ));
            })->afterResponse();
        }

        // Send admin email if enabled
        if ($setting->receive_buy_course_email) {
            dispatch(function () use ($setting, $user, $course_name) {
                Mail::to($setting->notifiable_email)->send(new MembershipMail([
                    'name' => $user->name,
                    'course_name' => $course_name,
                ]));
            })->afterResponse();
        }

        session()->flash('success', $response);
        $this->purchased = true;
    } catch (MembershipException $e) {
        session()->flash('message', $e->getMessage());
    } catch (\Throwable $e) {
        session()->flash('error', __('t.course_purchase_error'));
    }
}

}