<?php

namespace App\Services;

class ThemeColorService
{
    /**
     * Get the current theme colors from configuration
     */
    public function getThemeColors(): array
    {
        return config('theme-colors');
    }

    /**
     * Generate CSS custom properties from configuration
     */
    public function generateCssVariables(): string
    {
        $colors = $this->getThemeColors();
        $css = ":root {\n";

        // Primary colors
        $css .= "    --color-primary: {$colors['primary']['main']};\n";
        $css .= "    --color-primary-light: {$colors['primary']['light']};\n";
        $css .= "    --color-primary-dark: {$colors['primary']['dark']};\n";
        $css .= "    --color-primary-hover: {$colors['primary']['hover']};\n";

        // Primary with opacity
        foreach ($colors['opacity'] as $opacity => $value) {
            $rgb = $this->hexToRgb($colors['primary']['main']);
            $css .= "    --color-primary-{$opacity}: rgba({$rgb['r']}, {$rgb['g']}, {$rgb['b']}, {$value});\n";
        }

        // Background colors
        foreach ($colors['background'] as $name => $color) {
            $css .= "    --color-background-{$name}: {$color};\n";
        }

        // Text colors
        foreach ($colors['text'] as $name => $color) {
            $css .= "    --color-text-{$name}: {$color};\n";
        }

        // Border colors
        foreach ($colors['border'] as $name => $color) {
            $css .= "    --color-border-{$name}: {$color};\n";
        }

        // Status colors
        foreach ($colors['status'] as $name => $color) {
            $css .= "    --color-{$name}: {$color};\n";
        }

        $css .= "}\n";

        return $css;
    }

    /**
     * Apply a color scheme
     */
    public function applyColorScheme(string $schemeName): bool
    {
        $schemes = config('theme-colors.schemes');
        
        if (!isset($schemes[$schemeName])) {
            return false;
        }

        $scheme = $schemes[$schemeName];
        
        // Update the configuration (you might want to save this to database)
        config(['theme-colors.primary.main' => $scheme['primary']]);
        config(['theme-colors.primary.light' => $scheme['primary_light']]);
        config(['theme-colors.primary.dark' => $scheme['primary_dark']]);

        return true;
    }

    /**
     * Convert hex color to RGB array
     */
    private function hexToRgb(string $hex): array
    {
        $hex = ltrim($hex, '#');
        
        if (strlen($hex) === 3) {
            $hex = $hex[0] . $hex[0] . $hex[1] . $hex[1] . $hex[2] . $hex[2];
        }

        return [
            'r' => hexdec(substr($hex, 0, 2)),
            'g' => hexdec(substr($hex, 2, 2)),
            'b' => hexdec(substr($hex, 4, 2)),
        ];
    }

    /**
     * Get available color schemes
     */
    public function getAvailableSchemes(): array
    {
        return config('theme-colors.schemes');
    }

    /**
     * Update theme colors and save to configuration
     */
    public function updateThemeColors(array $colors): bool
    {
        try {
            // Validate colors
            foreach ($colors as $key => $value) {
                if (!$this->isValidHexColor($value)) {
                    throw new \InvalidArgumentException("Invalid color format: {$value}");
                }
            }

            // Update the config file content
            $configPath = config_path('theme-colors.php');
            $config = include $configPath;
            
            if (isset($colors['primary'])) {
                $config['primary']['main'] = $colors['primary'];
            }
            if (isset($colors['primary_light'])) {
                $config['primary']['light'] = $colors['primary_light'];
            }
            if (isset($colors['primary_dark'])) {
                $config['primary']['dark'] = $colors['primary_dark'];
            }
            if (isset($colors['background_secondary'])) {
                $config['background']['secondary'] = $colors['background_secondary'];
            }
            if (isset($colors['background_tertiary'])) {
                $config['background']['tertiary'] = $colors['background_tertiary'];
            }

            // Write back to config file
            $this->writeConfigFile($configPath, $config);
            
            // Clear config cache
            if (function_exists('config_clear')) {
                config_clear();
            }

            return true;
        } catch (\Exception $e) {
            \Log::error('Failed to update theme colors: ' . $e->getMessage());
            return false;
        }
    }

    /**
     * Write configuration array back to file
     */
    private function writeConfigFile(string $path, array $config): void
    {
        $content = "<?php\n\nreturn " . var_export($config, true) . ";\n";
        file_put_contents($path, $content);
    }

    /**
     * Validate hex color format
     */
    private function isValidHexColor(string $color): bool
    {
        return preg_match('/^#([A-Fa-f0-9]{6}|[A-Fa-f0-9]{3})$/', $color);
    }
}